clear variables; close all; clc
figure(1)
% ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
%            DIMENSIONAL ANALYSIS : calculation of
%                                   Péclet and Brinkman numbers
% ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

%                   VALUES
h                   = 5e3; % thickness of nappe [m]
ramp                = 15;  % ramp angle of the thrust [°]

T_0                 = 500 + 273.15; % initial temp at the depth zf [K]
rho_c               = 2700*1e3;     % Density * Heat Capacity [J/m^3*K]
k                   = 2.5;          % Thermal Conductivity [W/m*K]
k_diff              = k/rho_c;      % Thermal Diffusivity [m^2/s]

eta                 = [5e19 5e20 5e21 5e22];           % Viscosity [Pa*s]
vx_bg_v             = [0:0.1:200];                     % Velocity [cm/yr]
% horizontal, prescribed, background velocity [m/s]:
vx_bg               = vx_bg_v *1e-2 /(3600*24*365.25); 

% representative velocities for triplot [cm/yr]:
velocity = [0 0.5 1 2 3 5 10 20 50 100 200];

%==========================================================================

col = {'-g','-b','-r','-m'}; % set colours for different viscosities

%                   CALCULATIONS

A = ones(length(velocity),4,length(eta));

for ie = 1:length(eta)
    
    Pe(ie,:) = h*vx_bg*sind(ramp)/k_diff;% Peclet Number  (Adv vs Diff)
    Br(ie,:) = eta(ie).*vx_bg.^2/(k*T_0);% Brinkman Number(Prod_sh vs Diff)
        
    hold on
    plot(Pe(ie,:),Br(ie,:),col{ie},'linewidth',1.5) % 2D Pe-Br plot
    
    % plot points with same background velocity and store Pe, Br values:
            
        VALUES = ones(length(velocity),4);
        
        for iv=1:length(velocity)
            
            % find index of matrix vx_bg_v with value = matrix velocity:
            item(iv) = find(abs(vx_bg_v)==velocity(iv));
            % plot the corresponding point in Pe-Br plot:
            plot(Pe(ie,item(iv)),Br(ie,item(iv)),'ok','linewidth',1,...
                'HandleVisibility','off') 
            
            % plot isovelocity lines
            % Pe corresponding to vx_bg_v = velocity(iv):
            v_value_line = Pe(ie,item(iv)); 
            plot((v_value_line*(ones(1,length(vx_bg_v)))),Br(ie,:),':k',...
                'linewidth',1,'HandleVisibility','off')
            t = text(Pe(ie,item(iv)),1.65,['\leftarrow' ...
                num2str(vx_bg_v(item(iv))) ' cm/yr'])
            t.Rotation = 60;
            hold on
            
            % calculate new dimensionless parameters for ternary plot 
            % (Duprat-Oualid et al., 2015)
            R_adv (ie,:) = Pe(ie,item(iv)) /...   % Advection  contribution 
                (Pe(ie,item(iv))+1+Br(ie,item(iv)));  
            R_diff(ie,:) =  1 /...                % Diffusion  contribution
                (Pe(ie,item(iv))+1+Br(ie,item(iv)));
            R_pro (ie,:) = Br(ie,item(iv)) /...   % Production contribution
                (Pe(ie,item(iv))+1+Br(ie,item(iv)));
            values(ie,:) = [R_diff(ie,:) R_adv(ie,:) ...
                R_pro(ie,:) vx_bg_v(item(iv))];
            
            VALUES(iv,:) = values(ie,:);
            
        end
        
        VALUES_eta(:,:,ie) = VALUES(:,:);
                
end

%==========================================================================

% 2D plot
plot([0 1 1],[1 1 0],'k','linewidth',2,'HandleVisibility','off')
plot([1 2],[1 2],'k','linewidth',2,'HandleVisibility','off')
axis equal
title({'Dimensional Analysis'})
le=legend('5*10^{19} Pas','5*10^{20} Pas','5*10^{21} Pas','5*10^{22} Pas');
set(le,'position',[0.6192 0.3891 0.1782 0.1883])
xlabel('Péclet Number'),ylabel('Brinkman Number')
axis([0 2 0 2])
hold off

set(gcf, 'Position',  [[1032 409 664 541]])
stringg = ['print -dpng Dimensional_analysis','.png -r300'];eval(stringg);

%==========================================================================

% Ternary plot (modified after Theune, U. (2002-2005):
%  https://www.mathworks.com/matlabcentral/fileexchange/7210-ternary-plots
%  MATLAB Central File Exchange)

figure(2)
warning off MATLAB:griddata:DuplicateDataPoints

[h,hg,htick]=terplot; % Plot the ternary axis system
colormap(colorcube)
set(gca,'ColorScale','log')

for ie = 1:length(eta)
    A = VALUES_eta(:,:,ie); % diff - adv - pro - v(cm/yr)
    eta_new = eta(:,ie);
    [hd,hcb]=ternaryc(A(:,1),A(:,2),A(:,3),A(:,4),'o',eta_new); % Plot data
end

    hlabels=terlabel('R diff','R adv','R pro');
    hold on

set(gcf, 'Position',  [[1032 409 664 541]])
stringg = ['print -dpng Dimensional_analysis_triplot','.png -r300'];...
    eval(stringg);

